#!/usr/bin/perl

# all_resources.pl

# Use this script to index the complete list of resources including all metadata
# as well as term association. The output of this file is HTML and the indexer is
# intended to be something like Swish-E.

# 2006-01-12 - added pod - ELM
# first draft 8/09/2004 - RJF


=head1 NAME

all_resources.pl - dump all MyLibrary resources in a form indexable by swish-e

=head1 DESCRIPTION

This program loops through each resource in your MyLibrary database, formats each record as an HTML stream with very rich meta-data elements, adds a swish-e specific header to each stream, and sends the whole thing to STDOUT.

This program is intended to be run via another script (./index-resources.pl). The HTML's meta-data elements are expected to be echoed in a swish-e configuration file (./index-resources.cfg). These meta-data elements then become fields searchable in the resulting index.

=head1 SEE ALSO

index-resources.cfg and index-resources.pl

=head1 AUTHOR

Rob Fox

=cut

use lib '../lib/';
use MyLibrary::Resource;
use MyLibrary::Term;
use strict;

# first, get all of the resources
my @resource_ids = MyLibrary::Resource->get_ids();


foreach my $resource_id (@resource_ids) {
	my $resource = MyLibrary::Resource->new(id => $resource_id);
	my @related_terms = $resource->related_terms();
	my %facet_ids = ();
	my $output;
	foreach my $related_term_id (@related_terms) {
		my $term = MyLibrary::Term->new(id => $related_term_id);
		my $facet_id = $term->facet_id();
		if ($facet_ids{$facet_id}) {
			my @active_terms = @{$facet_ids{$facet_id}};
			push (@active_terms, $related_term_id);
			$facet_ids{$facet_id} = \@active_terms;
		} else {
			my @active_terms = ();
			push (@active_terms, $related_term_id);
			$facet_ids{$facet_id} = \@active_terms;
		}
	}
	$output = "<html>\n<head>\n\t<title>" . $resource->name() . "</title>\n";
	$output .= "\t<meta name=\"resource_id\" content=\"" . $resource->id() . "\" />\n";
	$output .= "\t<meta name=\"full_text\" content=\"" . $resource->full_text() . "\" />\n";
	$output .= "\t<meta name=\"reference_linking\" content=\"" . $resource->reference_linking() . "\" />\n";
	$output .= "\t<meta name=\"coverage_info\" content=\"" . $resource->coverage_info() . "\" />\n";
	$output .= "\t<meta name=\"description\" content=\"" . $resource->note() . "\" />\n";
	$output .= "\t<meta name=\"lcd\" content=\"" . $resource->lcd() . "\" />\n";
	
	# output locations
	my @resource_locations = $resource->resource_locations();
	foreach my $location (@resource_locations) {
		$output .= "\t<meta name=\"location\" content=\"" . $location->location() . "\" />\n";
	}

	# output facets and terms
	foreach my $facet_id (keys %facet_ids) {
		my $facet = MyLibrary::Facet->new(id => $facet_id);
		my $facet_name = $facet->facet_name();
		$facet_name =~ s/ /_/g;
		$facet_name =~ s/&/and/g;
		my @term_ids = @{$facet_ids{$facet_id}};
		foreach my $term_id (@term_ids) {
			my $term = MyLibrary::Term->new(id => $term_id);
			$output .= "\t<meta name=\"${facet_name}\" content=\"" . $term->term_name() . "\" />\n";
		}
	}

	$output .= "</head>\n<body>\n<pre>\n";
	$output .= "\t<h1>" . $resource->name() . "</h1>\n";
	$output .= "\t<p>" . $resource->note() . "</p>\n";
	$output .= "</pre>\n</body>\n</html>\n";	

	# output the record
	print "Path-Name: ${resource_id}\n";
	print "Content-length: " . scalar(length $output) . "\n";
	print "Document-Type: HTML*\n";
	print "\n";

	print "$output";
}
