#!/usr/bin/perl

# location-types.cgi - create and maintain MyLibrary location types

# Eric Lease Morgan <emorgan@nd.edu>
# January 27, 2006 - moving to the MyLibrary distro


=head1 NAME

location-types.cgi - create and manage types of information locations


=head1 DESCRIPTION

Use this script to create and manage information types in your MyLibrary implementation.

Information often times has a location. Books have call numbers. Internet resources have URL's. Some Internet resources have multiple URL's. One for the resource itself and another pointing to a help file or scope note. Microfiche might be located in cabinets. Etc. This program allows you to create these location types, and each MyLibrary implementation will need at least one location type. You will probably want to create a location type called something like Internet Resource or URL.

This program requires no configuration.


=head1 AUTHOR

Eric Lease Morgan


=cut


# require the necessary modules
use lib '../lib';
use CGI qw/:standard/;
use CGI::Carp qw(fatalsToBrowser);
use strict;
use MyLibrary::Core;
require 'etc/subroutines.pl';

# initialize two global variables
my $cgi = CGI->new;
my $html = '';

# branch according to the input
if (! $cgi->param('cmd')) { &home }
elsif ($cgi->param('cmd') eq 'create')  { &create }
elsif ($cgi->param('cmd') eq 'find')    { &find }
elsif ($cgi->param('cmd') eq 'edit')    { &edit }
elsif ($cgi->param('cmd') eq 'delete')  { &delete }
else { $html = $cgi->p("Unknown command. Call Eric.") }

# quit
&gracefulExit;


#############
# subroutines

sub header {
	
	# get the value of this script
	my $script = $cgi->script_name;

	# return a standard html page; notice the stylesheet
	return <<EOH;
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
<html xmlns="http://www.w3.org/1999/xhtml">
	<head>
		<title>Location types</title>
		<link rel='stylesheet' href='../etc/portal-screen.css' type='text/css' />
	</head>
	<body>
		<div id="navcontainer">
			<ul id="navlist">
				<li><a href="admin.cgi">Home</a></li>
				<li><a href="facets.cgi">Facets</a></li>
				<li><a href="terms.cgi">Terms</a></li>
				<li><a href="location-types.cgi" id="current">Location types</a></li>
				<li><a href="resources.cgi">Resources</a></li>
				<li><a href="patrons.cgi">Patrons</a></li>
			</ul>
		</div>
	
	<div class='menu'>
		<p><b><a href="$script">Manage location types</a></b></p>
		<a href="$script?cmd=create">Create</a><br />
		<a href="$script?cmd=find">Find all</a><br />
		<a href="$script?cmd=edit">Edit</a><br />
		<a href="$script?cmd=delete">Delete</a><br />
		<br />
	</div>
	<div class='content'>
EOH

}


sub home {

	# build a simple home page
	$html  = $cgi->h1('Manage location types');
	$html .= $cgi->p('Use these pages to manage MyLibrary Portal location types.');

}


sub create {

	# initialize the output
	$html = $cgi->h1('Create');
	
	# branch according to the input
	if (! $cgi->param('button')) {
		
		# build the form
		$html .= $cgi->p('Use this form to create a location type.');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'create');
		$html .= table({-border => 0}, Tr({-align => 'left', -valign => 'top'},
						[
							th(['Item', 'Value']),
							td(['Location type', $cgi->textfield(-name => 'name')]),
							td(['Note', $cgi->textarea(-name => 'description', -rows => 7, -columns => 40)]),
							td([''    , $cgi->submit(-name => 'button', -value => 'Create')])
						]
					)
				 );
		$html .= $cgi->endform;
    
    }
    
    else {
    
    	# create an object, fill it, and save
    	my $location_type = MyLibrary::Resource::Location::Type->new;
    	$location_type->name($cgi->param('name'));
    	$location_type->description($cgi->param('description'));
		$location_type->commit;
				
		# echo the results
		$html .= $cgi->p('Location type created:');
		$html .= $cgi->ul(li('ID -', $location_type->location_type_id),
		                  li('Location type - ', $location_type->name),
		                  li('Note - ', $location_type->description)
		                 );
    
    }
    
}


sub find {

	# initialize the output
	$html  = $cgi->h1('All location types');
	$html .= $cgi->p('This is a list of all the location types:');
	
	# get all the facets, build a list, and display it
	my @location_type_ids = MyLibrary::Resource::Location::Type->all_types;
	my $items;
	foreach (@location_type_ids) {
	
		my $location_type = MyLibrary::Resource::Location::Type->new(id => $_);
		$items .= $cgi->li($location_type->name, ' (' . $location_type->location_type_id . ') - ', $location_type->description);
		
	}
	$html .= $cgi->ul($items);
	
}


sub edit {

	# initialize the output and a button
	$html = $cgi->h1('Edit location types');
	my $submit .= $cgi->submit(-name => 'button', -value => 'Edit');
	
	# branch according to the input
	if (! $cgi->param('button')) {
	
		# create a hash of location types; needs to be sorted
		my @location_types_ids = MyLibrary::Resource::Location::Type->all_types;
		my %location_types;
		foreach (@location_types_ids) {
		
			my $location_type = MyLibrary::Resource::Location::Type->new(id => $_);
			$location_types{$location_type->location_type_id} = $location_type->name;
		
		}
	
		# display a form
		$html .= $cgi->p('Select a location type to edit');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'edit');
		$html .= $cgi->popup_menu(-name => 'id', -values => \%location_types);
		$html .= $submit;			 
		$html .= $cgi->endform;

	}
	
	elsif (! $cgi->param('name') || ! $cgi->param('description')) {
	
    	# create an object based on the input, and create form parts
    	my $location_type = MyLibrary::Resource::Location::Type->new(id => $cgi->param('id'));
    	
		# display an edit form
		$html .= $cgi->p('Use this form to edit a location type.');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'edit');
		$html .= $cgi->hidden(-name => 'id', -value => $location_type->location_type_id);
		$html .= table({-border => 0},
					Tr({-align => 'left', -valign => 'top'},
						[
							th(['Item',  'Value']),
							td(['ID'  ,  $location_type->location_type_id]),
							td(['Location type', $cgi->textfield(-name => 'name', -value => $location_type->name)]),
							td(['Note',  $cgi->textarea(-name => 'description', -value => $location_type->description, -rows => 7, -columns => 40)]),
							td([''    ,  $submit])
						]
					)
				 );
		$html .= $cgi->endform;
    
    }
    
	else {

		# create and object, fill it, and save
		my $location_type = MyLibrary::Resource::Location::Type->new(id => $cgi->param('id'));
		$location_type->name($cgi->param('name'));
		$location_type->description($cgi->param('description'));
		$location_type->commit;
		
		# echo the result
		$html .= $cgi->p('Location type updated.');
		$html .= $cgi->p('ID:', $location_type->location_type_id, '<br />Location type:', $location_type->name, '<br />Note:' , $location_type->description);
    
    }

}


sub delete {

	# initialize the output
	$html = $cgi->h1('Delete location types');
	
	# branch according to the input
	if (! $cgi->param('button')) {
	
		# create a hash of location types; needs to be sorted
		my @location_types_ids = MyLibrary::Resource::Location::Type->all_types;
		my %location_types;
		foreach (@location_types_ids) {
		
			my $location_type = MyLibrary::Resource::Location::Type->new(id => $_);
			$location_types{$location_type->location_type_id} = $location_type->name;
		
		}
	
		# display a pop-up list of facets
		$html .= $cgi->p('Select a location type to delete:');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'delete');
		$html .= $cgi->popup_menu(-name => 'id', -values => \%location_types);
		$html .= $cgi->submit(-name => 'button', -value => 'delete', -label => 'Delete');		 
		$html .= $cgi->endform;

	}
	
	elsif ($cgi->param('button') eq 'Delete') {
    
    	# create an object based on input
    	my $location_type = MyLibrary::Resource::Location::Type->new(id => $cgi->param('id'));
    	
		# display a confirmation
		$html .= $cgi->p('Are you sure you want to delete the location type named <b>' . $location_type->name . '</b>?');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'delete');
		$html .= $cgi->hidden(-name => 'id', -value => $location_type->location_type_id);
		$html .= $cgi->submit(-name => 'button', -value => 'yes', -label => 'Yes');
		$html .= ' ';
		$html .= $cgi->submit(-name => 'button', -value => 'no', -label => 'No');		 
		$html .= $cgi->endform;
    
    }
    
    elsif ($cgi->param('button') eq 'Yes') {
    
    	# delete the facet; do the work and echo the result
    	my $location_type = MyLibrary::Resource::Location::Type->new(id => $cgi->param('id'));
    	$location_type->delete;
    	$html .= $cgi->p('The location type ' . $location_type->name . ' has been deleted.');
    
    }
    
    elsif ($cgi->param('button') eq 'No') {
    
    	# abort
    	my $location_type = MyLibrary::Resource::Location::Type->new(id => $cgi->param('id'));
    	$html .= $cgi->p('The location type ' . $location_type->name . ' has <b>not</b> been deleted.');

    }
    
}


sub gracefulExit {

	# output the 'magic line', a standard header, the content, and a standard footer, then done
	print $cgi->header;
	print &header;
	print $html;
	print &footer;
	exit;

}

